/* $Id: raw2mq.c,v 1.6 2025/04/21 11:07:18 urabe Exp $ */
/* "raw2mq.c"   2023.2.5 urabe */
/*              modified from raw_ch.c */
/*              2023.2.5-2.19 raw2mq.c */
/*              2023.2.23 daemon mode */
/*              2025.4.13 JSON mode(-j)/ASCII mode(-a) with per-ch topics */
/*              2025.4.21 options -l/-L/-n for -j/-a  */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <syslog.h>
#include <math.h>
#include <mosquitto.h>

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else  /* !TIME_WITH_SYS_TIME */
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else  /* !HAVE_SYS_TIME_H */
#include <time.h>
#endif  /* !HAVE_SYS_TIME_H */
#endif  /* !TIME_WITH_SYS_TIME */

#include "winlib.h"
#include "daemon_mode.h"
#define PORT 1883
#define SPORT 8883
#define CAFILE "/etc/ssl/cert.pem"
#define CAPATH "/etc/ssl/certs"
#define RAW   0
#define JSON  1
#define ASCII 2

static const char rcsid[] =
  "$Id: raw2mq.c,v 1.6 2025/04/21 11:07:18 urabe Exp $";

char *progname,*logfile;
static int daemon_mode;
int syslog_mode,exit_status;
int debug=0;
int connected=0;
char tb[256];
struct mosquitto *mosq;
time_t ltime_p;
unsigned long n_packets,n_bytes;   /* 64bit ok */
int abuf[4095]; /* fix data buffer for one ch */
uint8_w cbuf[4095*10]; /* ascii data buffer for one ch */
struct {
  int flag;
  char name[32];
  double upb; /* unit(m/s, m/s/s etc.) per bit */
} chtbl[WIN_CHMAX];
int read_ch=0; /* 0:none, 1: read, 2: read and phys conv */

void
report()
{
  time_t ltime,j_timt,k_timt;

  time(&ltime);
  j_timt=ltime-ltime_p;
  if(j_timt != 0) {
    k_timt=j_timt/2;
    if(ltime_p)
      {
      snprintf(tb,sizeof(tb),
	"%lu msgs %lu B in %ld s ( %ld msgs/s %ld B/s ) ",
	 n_packets,n_bytes,(long)j_timt,
	 (n_packets+k_timt)/j_timt,(n_bytes+k_timt)/j_timt);
      write_log(tb);
      n_packets=n_bytes=0;
      }
    ltime_p=ltime;
  }
}

void
endprog()
{
  mosquitto_destroy(mosq);
  mosquitto_lib_cleanup();
  end_program();
}

void
usage(void)
{
  WIN_version();
  fprintf(stderr, "%s\n", rcsid);
  if(daemon_mode)
    fprintf(stderr,
  " usage : '%s (-agjns) (-t [topic]) (-p [port]) (-q [qos]) (-i [id])\\\n\
               (-c [cafile] / -d [capath]) (-C [certfile] -k [keyfile])\\\n\
               (-u [user]  -P [pswd] -l/-L [chtbl])\\\n\
             [in_key] [host] ([log file])'\n",progname);
  else
    fprintf(stderr,
  " usage : '%s (-agjnsD) (-t [topic]) (-p [port]) (-q [qos]) (-i [id])\\\n\
               (-c [cafile] / -d [capath]) (-C [certfile] -k [keyfile])\\\n\
               (-u [user]  -P [pswd] -l/-L [chtbl])\\\n\
             [in_key] [host] ([log file])'\n",progname);
}

void
on_connect(struct mosquitto *mosq,void *obj,int rc)
{
  if(rc) {
    snprintf(tb,sizeof(tb),"%s",mosquitto_connack_string(rc));
    write_log(tb);
    endprog();
  }
  connected=1;
  write_log("connected");
}

void
on_disconnect(struct mosquitto *mosq,void *obj,int rc)
{
  connected=0;
  write_log("disconnected");
}

int
fix2json(uint8_w *ts,int *abuf,WIN_ch ch,WIN_sr sr,uint8_w *cbuf,char *chs)
{
  uint8_w *p;
  int i;
  p=cbuf;
  p+=sprintf(p,
    "{\"t\":[%X,%X,%X,%X,%X,%X],\"n\":1,\"ch%s\":{\"f\":%d,\"d\":[",
    ts[0],ts[1],ts[2],ts[3],ts[4],ts[5],chs,sr);
  if(read_ch==2) {
    for(i=0;i<sr-1;i++) p+=sprintf(p,"%g,",(double)abuf[i]*chtbl[ch].upb);  
    p+=sprintf(p,"%g]},\"chs\":[\"%s\"]}",(double)abuf[sr-1]*chtbl[ch].upb,chs);
  }
  else {
    for(i=0;i<sr-1;i++) p+=sprintf(p,"%d,",abuf[i]);  
    p+=sprintf(p,"%d]},\"chs\":[\"%s\"]}",abuf[sr-1],chs);
  }
  return p-cbuf;
}

int
fix2ascii(uint8_w *ts,int *abuf,WIN_ch ch,WIN_sr sr,uint8_w *cbuf,char *chs)
{
  uint8_w *p;
  int i;
  p=cbuf;
  p+=sprintf(p,"%02X %02X %02X %02X %02X %02X 1\n%s %d",
    ts[0],ts[1],ts[2],ts[3],ts[4],ts[5],chs,sr);
  if(read_ch==2)
    for(i=0;i<sr;i++) p+=sprintf(p," %g",(double)abuf[i]*chtbl[ch].upb);
  else
    for(i=0;i<sr;i++) p+=sprintf(p," %d",abuf[i]);
  return p-cbuf;
}

int
main(int argc, char *argv[])
{
  struct Shm *shr;
  key_t rawkey;
  uint32_w uni;
  uint8_w *ptr,*ptr_lim,*ptr_save,ts[6],*ptw;
  WIN_ch ch;
  WIN_sr sr;
  int c,i,tow,ret,rc,msize;
  int eobsize_in, eobsize_in_count;
  uint32_w size;
  unsigned long c_save;
  uint16_t port=PORT;
  char *host=NULL,*id=NULL,*user=NULL,*pswd=NULL,*capath=NULL,*certfile=NULL,
    *keyfile=NULL,*cafile=NULL,*topic="win/data",*topic1,topic2[100];
  int qos=0;
  int ssl=0;
  unsigned int mid=0;
  int portset=0;
  int ca_default=1;
  int outmode=RAW;
  n_packets=n_bytes=0;
  int tpc=1; /* 0:CHID, 1:STN/COMP */
  FILE *fp;
  char *chfile,stan[16],comp[16],*tp1,tp2[32];
  double sens,gain,lsb;

  if((progname=strrchr(argv[0],'/')) != NULL) progname++;
  else progname=argv[0];

  daemon_mode=syslog_mode=0;
  exit_status=EXIT_SUCCESS;
  if(strcmp(progname,"raw2mqd")==0) daemon_mode=1;

  while((c=getopt(argc,argv,"ac:C:d:Dgi:jk:l:L:np:P:q:st:u:"))!=-1)
    {
    switch(c)
      {
      case 'a':   /* ASCII output mode */
        outmode|=ASCII;
        break;
      case 'c':   /* cafile */
        cafile=optarg;
        ca_default=0;
        break;
      case 'C':   /* certfile */
        certfile=optarg;
        break;
      case 'd':   /* capath */
        capath=optarg;
        ca_default=0;
        break;
      case 'D':   /* daemon mode */
        daemon_mode = 1;
        break;
      case 'g':   /* debug print */
        debug++;
        break;
      case 'i':   /* unique client id */
        id=optarg;
        break;
      case 'j':   /* JSON output mode */
        outmode|=JSON;
        break;
      case 'k':   /* keyfile */
        keyfile=optarg;
        break;
      case 'l':   /* chtbl */
        chfile=optarg;
        read_ch=1;
        break;
      case 'L':   /* chtbl */
        chfile=optarg;
        read_ch=2;
        break;
      case 'n':   /* use CHID in topic */
        tpc=0;
        break;
      case 'p':   /* port */
        port=(uint16_t)atoi(optarg);
        portset=1;
        break;
      case 'P':   /* password */
        pswd=optarg;
        break;
      case 'q':   /* qos */
        qos=atoi(optarg);
        break;
      case 's':   /* use ssl/tls */
        ssl=1;
        break;
      case 't':   /* topic */
        topic=optarg;
        break;
      case 'u':   /* username */
        user=optarg;
        break;
      default:
        fprintf(stderr," option -%c unknown\n",c);
        usage();
        exit(1);
      }
    }
  optind--;
  if(argc<3+optind)
    {
    usage();
    exit(1);
    }
  rawkey=atol(argv[1+optind]);
  host=argv[2+optind];
  if(argc>3+optind) logfile=argv[3+optind];
  else {
    logfile=NULL;
    if(daemon_mode) syslog_mode=1;
  }

  /* daemon mode */
  if (daemon_mode) {
    daemon_init(progname, LOG_USER, syslog_mode);
    umask(022);
  }

  if(qos<0 || qos>2) {
    write_log("invalid QoS value");
    exit(1);
  }

  if(ssl && !portset) port=SPORT;

  write_log("start");

  if(*topic==0 || *topic==' ') {
    if(outmode==RAW) {
      write_log("invalid topic");
      exit(1);
    }
    *topic1=0;  
  } else topic1=topic;

  if(read_ch) {
    if(outmode==RAW) write_log("option -l/-L ignored");
    else {
      if((fp=fopen(chfile,"r"))==NULL) {
        write_log("chtbl not open!");
        exit(1);
      }
      for(i=0;i<WIN_CHMAX;i++) chtbl[i].flag=0;
      i=0;
      while(fgets(tb,sizeof(tb),fp)) { /* read each line */
        if(*tb=='#') continue;
        if(sscanf(tb,"%x%*s%*s%s%s%*s%*s%lf%*s%*s%*s%lf%lf",
            &ch,stan,comp,&sens,&gain,&lsb)<1)
            /* sens=V/unit, lsb=V/LSB, gain(dB) */
          continue;  /* skip blank line */
        chtbl[ch].flag=1;
        sprintf(chtbl[ch].name,"%s/%s",stan,comp);
        chtbl[ch].upb=(lsb/sens)/pow(10.0,(gain/20.0));
        i++;
        if(debug) printf("i=%d ch=%04X stan=%s comp=%s name=%s sens=%g gain=%g lsb=%g upb=%g\n",
          i,ch,stan,comp,chtbl[ch].name,sens,gain,lsb,chtbl[ch].upb);
      }
      fclose(fp);
      snprintf(tb,sizeof(tb),"%d chs read from %s",i,chfile);
      write_log(tb);
    }
  } else tpc=0;

  if(outmode&JSON) {
    if(tpc==0) strcpy(tp2,"CHID");
    else strcpy(tp2,"STN/CMP");
    if(read_ch==2) tp1="J";
    else tp1="j";
    if(*topic1==0) snprintf(topic2,sizeof(topic2),"%s/%s",tp1,tp2);
    else snprintf(topic2,sizeof(topic2),"%s/%s/%s",topic1,tp1,tp2);
    snprintf(tb,sizeof(tb),"JSON format (with topics %s)",topic2);
    write_log(tb);
  }
  if(outmode&ASCII) {
    if(tpc==0) strcpy(tp2,"CHID");
    else strcpy(tp2,"STN/CMP");
    if(read_ch==2) tp1="A";
    else tp1="a";
    if(*topic1==0) snprintf(topic2,sizeof(topic2),"%s/%s",tp1,tp2);
    else snprintf(topic2,sizeof(topic2),"%s/%s/%s",topic1,tp1,tp2);
    snprintf(tb,sizeof(tb),"ASCII format (with topics %s)",topic2);
    write_log(tb);
  }
  if(outmode==RAW) {
    sprintf(tb,"topic=%s",topic1);
    write_log(tb);
  }

  mosquitto_lib_init();
  if(id==NULL) mosq=mosquitto_new(NULL,1,NULL);
  else mosq=mosquitto_new(id,0,NULL);
  if(!mosq) {
    write_log("failed to create mosquitto");
    endprog();
  }

  mosquitto_username_pw_set(mosq,user,pswd);

  if(ssl) {
    if(ca_default) {
      rc=mosquitto_tls_set(mosq,cafile=CAFILE,capath,certfile,keyfile,NULL);
      if(rc!=MOSQ_ERR_SUCCESS) {
        rc=mosquitto_tls_set(mosq,cafile=NULL,capath=CAPATH,certfile,keyfile,NULL);
        if(rc!=MOSQ_ERR_SUCCESS) {
          snprintf(tb,sizeof(tb),"mosquitto_tls_set failed(%d)\n",rc);
          write_log(tb);
          endprog();
        }
      }
    } else {
      rc=mosquitto_tls_set(mosq,cafile,capath,certfile,keyfile,NULL);
      if(rc!=MOSQ_ERR_SUCCESS) {
        snprintf(tb,sizeof(tb),"mosquitto_tls_set failed(%d)\n",rc);
        write_log(tb);
        endprog();
      }
    }
  }

  if(ssl) snprintf(tb,sizeof(tb),"mqtts://%s:%s@%s:%d topic=[TOPIC] qos=%d",
    user,pswd,host,port,qos);
  else snprintf(tb,sizeof(tb),"mqtt://%s:%s@%s:%d topic=[TOPIC] qos=%d",
    user,pswd,host,port,qos);
  write_log(tb);
  if(ssl) {
    if(cafile) {snprintf(tb,sizeof(tb),"cafile=%s",cafile);write_log(tb);}
    if(capath) {snprintf(tb,sizeof(tb),"capath=%s",capath);write_log(tb);}
    if(certfile) {snprintf(tb,sizeof(tb),"certfile=%s",certfile);write_log(tb);}
    if(keyfile) {snprintf(tb,sizeof(tb),"keyfile=%s",keyfile);write_log(tb);}
  }
  if(id) {snprintf(tb,sizeof(tb),"id=%s",id);write_log(tb);}

  mosquitto_connect_callback_set(mosq,on_connect);
  mosquitto_disconnect_callback_set(mosq,on_disconnect);

  if(mosquitto_connect_bind(mosq,host,port,60,NULL)) {
    strerror_r(errno,tb,sizeof(tb));
    write_log("failed to connect broker");
    endprog();
  }
  if(mosquitto_loop_start(mosq)!=MOSQ_ERR_SUCCESS){
    write_log("Unable to start loop");
    endprog();
  }
  sleep(1);

  /* shared memory */
  shr=Shm_read(rawkey,"in");

  signal(SIGTERM,(void *)endprog);
  signal(SIGINT,(void *)endprog);
  signal(SIGHUP,(void *)report);
  time(&ltime_p);

reset:
  while(shr->r==(-1)) sleep(1);
  ptr=shr->d+shr->r;
  tow=(-1);

  size=mkuint4(ptr);
  if(mkuint4(ptr+size-4)==size) eobsize_in=1;
  else eobsize_in=0;
  eobsize_in_count=eobsize_in;
  snprintf(tb,sizeof(tb),"eobsize_in=%d",eobsize_in);
  write_log(tb);

  for(;;) {
    size=mkuint4(ptr_save=ptr);
    if(mkuint4(ptr+size-4)==size) {
      if(++eobsize_in_count==0) eobsize_in_count=1;
    }
    else eobsize_in_count=0;
    if(eobsize_in && eobsize_in_count==0) goto reset;
    if(!eobsize_in && eobsize_in_count>3) goto reset;
    ptr_lim=ptr+size;
    if(eobsize_in) ptr_lim-=4;
    if(debug>1) {
      printf("    ");
      for(i=0;i<16;i++) printf("%02X",ptr[i]);
      printf(" :%6d b read\n",size);
    }

    c_save=shr->c;
    ptr+=4;
  /* make output data */
    uni=(uint32_w)(time(NULL)-TIME_OFFSET);
    i=uni-mkuint4(ptr);
    if(i>=0 && i<1440)   /* with tow */
      {
      if(tow!=1)
        {
	snprintf(tb,sizeof(tb),"with TOW (diff=%ds)",i);
        write_log(tb);
        if(tow==0)
          {
          write_log("reset");
          goto reset;
          }
        tow=1;
        }
      ptr+=4;
      }
    else if(tow!=0)
      {
      write_log("without TOW");
      if(tow==1)
        {
        write_log("reset");
        goto reset;
        }
      tow=0;
      }

    if(connected) { /* ptr -> ptr_lim */
      if(outmode!=RAW) {
        memcpy(ts,ptr,6); /* TS */
        ptr+=6;
        while(ptr<ptr_lim) {
          ptr+=win2fix(ptr,abuf,&ch,&sr);
          if(read_ch>0 && chtbl[ch].flag==0) continue;
          if(outmode&JSON) {
            if(tpc==0) sprintf(tp2,"%04X",ch);
            else sprintf(tp2,"%s",chtbl[ch].name);
            msize=fix2json(ts,abuf,ch,sr,cbuf,tp2);
            if(read_ch==2) tp1="J";
            else tp1="j";
            if(*topic1==0) snprintf(topic2,sizeof(topic2),"%s/%s",tp1,tp2);
            else snprintf(topic2,sizeof(topic2),"%s/%s/%s",topic1,tp1,tp2);
            ret=mosquitto_publish(mosq,&mid,topic2,msize,cbuf,qos,0);
            if(ret!=MOSQ_ERR_SUCCESS) {
              write_log("failed to publish");
            }
            if(debug) {
              for(i=0;i<36;i++) if(cbuf[i]=='\n') cbuf[i]=' ';
              printf("%.36s",cbuf);
              printf(".. %4d b sent topic=%s qos=%d mid=%d\n",
                msize,topic2,qos,mid);
            }
            mid++;
            n_packets++;
            n_bytes+=msize;
          }
          if(outmode&ASCII) {
            if(tpc==0) sprintf(tp2,"%04X",ch);
            else sprintf(tp2,"%s",chtbl[ch].name);
            msize=fix2ascii(ts,abuf,ch,sr,cbuf,tp2);
            if(read_ch==2) tp1="A";
            else tp1="a";
            if(*topic1==0) snprintf(topic2,sizeof(topic2),"%s/%s",tp1,tp2);
            else snprintf(topic2,sizeof(topic2),"%s/%s/%s",topic1,tp1,tp2);
            ret=mosquitto_publish(mosq,&mid,topic2,msize,cbuf,qos,0);
            if(ret!=MOSQ_ERR_SUCCESS) {
              write_log("failed to publish");
            }
            if(debug) {
              for(i=0;i<36;i++) if(cbuf[i]=='\n') cbuf[i]=' ';
              printf("%.36s",cbuf);
              printf(".. %4d b sent topic=%s qos=%d mid=%d\n",
                msize,topic2,qos,mid);
            }
            mid++;
            n_packets++;
            n_bytes+=msize;
          }
        }
      } else { /* WIN */
        ret=mosquitto_publish(mosq,&mid,topic1,ptr_lim-ptr,ptr,qos,0);
        if(ret!=MOSQ_ERR_SUCCESS) {
          write_log("failed to publish");
        }
        if(debug) {
          for(i=0;i<16;i++) printf("%02X",ptr[i]);
          printf(".. %6d b sent topic=%s qos=%d mid=%d\n",
            ptr_lim-ptr,topic1,qos,mid);
        }
        mid++;
        n_packets++;
        n_bytes+=ptr_lim-ptr;
      }
    }
    if((ptr=ptr_save+size)>shr->d+shr->pl) ptr=shr->d;
    while(ptr==shr->d+shr->p) usleep(10000);
    if(shr->c<c_save || mkuint4(ptr_save)!=size) {
      write_log("reset");
      goto reset;
    }
  }
}
